<?php

namespace App\Extensions\MarketingBot\System\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ShopifyProduct extends Model
{
    protected $table = 'ext_shopify_products';

    protected $fillable = [
        'shopify_store_id',
        'shopify_product_id',
        'title',
        'description',
        'handle',
        'product_type',
        'vendor',
        'price',
        'compare_at_price',
        'inventory_quantity',
        'inventory_policy',
        'image_url',
        'images',
        'variants',
        'status',
        'is_active',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'compare_at_price' => 'decimal:2',
        'inventory_quantity' => 'integer',
        'images' => 'array',
        'variants' => 'array',
        'is_active' => 'boolean',
    ];

    /**
     * Get the store that owns this product
     */
    public function store(): BelongsTo
    {
        return $this->belongsTo(ShopifyStore::class, 'shopify_store_id');
    }

    /**
     * Check if product is in stock
     */
    public function inStock(): bool
    {
        return $this->inventory_quantity > 0;
    }

    /**
     * Check if product is low stock
     */
    public function isLowStock(int $threshold = 10): bool
    {
        return $this->inventory_quantity > 0 && $this->inventory_quantity <= $threshold;
    }

    /**
     * Check if product is on sale
     */
    public function isOnSale(): bool
    {
        return $this->compare_at_price && $this->compare_at_price > $this->price;
    }

    /**
     * Get discount percentage
     */
    public function getDiscountPercentage(): ?float
    {
        if (!$this->isOnSale()) {
            return null;
        }

        return round((($this->compare_at_price - $this->price) / $this->compare_at_price) * 100, 2);
    }

    /**
     * Get product URL
     */
    public function getProductUrl(): string
    {
        $shopDomain = $this->store->shop_domain;
        return "https://{$shopDomain}/products/{$this->handle}";
    }
}
